"use client"

import type React from "react"

import Link from "next/link"
import { usePathname, useRouter } from "next/navigation"
import { useEffect, useState } from "react"

const navItems = [
  { name: "Home", href: "/" },
  { name: "Portfolio", href: "/portfolio" },
  { name: "Guides", href: "/guides" },
  { name: "Docs", href: "/docs" },
]

export function SiteHeader() {
  const pathname = usePathname()
  const router = useRouter()
  const [isAdmin, setIsAdmin] = useState(false)
  const [isLoggedIn, setIsLoggedIn] = useState(false)
  const [canAccessDashboard, setCanAccessDashboard] = useState(false)

  useEffect(() => {
    const currentUser = localStorage.getItem("currentUser")
    if (currentUser) {
      const userData = JSON.parse(currentUser)
      setIsAdmin(userData.role === "admin")
      setIsLoggedIn(true)
      setCanAccessDashboard(userData.role === "admin" || userData.role === "moderator")
    } else {
      setIsLoggedIn(false)
      setIsAdmin(false)
      setCanAccessDashboard(false)
    }
  }, [pathname])

  const handleDashboardClick = (e: React.MouseEvent) => {
    e.preventDefault()
    const currentUser = localStorage.getItem("currentUser")
    if (currentUser) {
      const userData = JSON.parse(currentUser)
      if (userData.role === "admin") {
        router.push("/admin")
      } else {
        router.push("/dashboard")
      }
    }
  }

  return (
    <header className="border-b border-border/50 backdrop-blur-sm bg-background/80 sticky top-0 z-50">
      <div className="container mx-auto flex items-center justify-between px-6 py-4">
        <Link href="/" className="flex items-center gap-3">
          <div
            className="flex h-10 w-10 items-center justify-center rounded text-black font-bold text-xl shadow-sm"
            style={{ background: "#b7ff00" }}
          >
            FS
          </div>
          <div>
            <h1 className="text-lg font-bold leading-none">Fuze Studios</h1>
          </div>
        </Link>

        {/* Navigation */}
        <nav className="hidden md:flex items-center gap-2">
          {navItems.map((item) => {
            const isActive = pathname === item.href
            return (
              <Link
                key={item.href}
                href={item.href}
                className={`px-4 py-2 text-sm font-medium rounded-lg transition-colors border-2 ${
                  isActive
                    ? "text-black border-transparent"
                    : "text-muted-foreground border-transparent hover:border-[#b7ff00] hover:text-foreground"
                }`}
                style={isActive ? { background: "#b7ff00" } : {}}
              >
                {item.name}
              </Link>
            )
          })}
        </nav>

        <div className="flex items-center gap-3">
          {canAccessDashboard && (
            <button
              onClick={handleDashboardClick}
              className={`flex items-center gap-2 px-5 py-2 rounded-full text-black font-medium shadow-sm border-2 border-transparent hover:bg-transparent hover:border-[#b7ff00] hover:text-[#b7ff00] transition-colors text-sm ${
                pathname === "/dashboard" || pathname === "/admin"
                  ? "ring-2 ring-[#b7ff00] ring-offset-2 ring-offset-black"
                  : ""
              }`}
              style={{ background: "#b7ff00" }}
            >
              <svg
                className="w-4 h-4"
                viewBox="0 0 24 24"
                fill="none"
                stroke="currentColor"
                strokeWidth="2"
                strokeLinecap="round"
                strokeLinejoin="round"
              >
                <rect x="3" y="3" width="7" height="7"></rect>
                <rect x="14" y="3" width="7" height="7"></rect>
                <rect x="14" y="14" width="7" height="7"></rect>
                <rect x="3" y="14" width="7" height="7"></rect>
              </svg>
              Dashboard
            </button>
          )}
          <a
            href="https://discord.gg/fuze"
            target="_blank"
            rel="noopener noreferrer"
            className="flex items-center gap-2 px-5 py-2 rounded-full text-black font-medium shadow-sm border-2 border-transparent hover:bg-transparent hover:border-[#b7ff00] hover:text-[#b7ff00] transition-colors text-sm"
            style={{ background: "#b7ff00" }}
          >
            <svg className="w-4 h-4" viewBox="0 0 24 24" fill="currentColor">
              <path d="M20.317 4.37a19.791 19.791 0 0 0-4.885-1.515a.074.074 0 0 0-.079.037c-.21.375-.444.864-.608 1.25a18.27 18.27 0 0 0-5.487 0a12.64 12.64 0 0 0-.617-1.25a.077.077 0 0 0-.079-.037A19.736 19.736 0 0 0 3.677 4.37a.07.07 0 0 0-.032.027C.533 9.046-.32 13.58.099 18.057a.082.082 0 0 0 .031.057a19.9 19.9 0 0 0 5.993 3.03a.078.078 0 0 0 .084.028a14.09 14.09 0 0 0 1.226-1.994a.077.077 0 0 0-.041-.106a13.107 13.107 0 0 1-1.872-.892a.077.077 0 0 1-.008-.128a10.2 10.2 0 0 0 .372-.292a.074.074 0 0 1 .077-.01c3.928 1.793 8.18 1.793 12.062 0a.074.074 0 0 1 .078.01c.12.098.246.198.373.292a.077.077 0 0 1-.006.127a12.299 12.299 0 0 1-1.873.892a.077.077 0 0 0-.041.107c.36.698.772 1.362 1.225 1.993a.076.076 0 0 0 .084.028a19.839 19.839 0 0 0 6.002-3.03a.077.077 0 0 0 .032-.054c.5-5.177-.838-9.674-3.549-13.66a.061.061 0 0 0-.031-.03zM8.02 15.33c-1.183 0-2.157-1.085-2.157-2.419c0-1.333.956-2.419 2.157-2.419c1.21 0 2.176 1.096 2.157 2.42c0 1.333-.956 2.418-2.157 2.418zm7.975 0c-1.183 0-2.157-1.085-2.157-2.419c0-1.333.955-2.419 2.157-2.419c1.21 0 2.176 1.096 2.157 2.42c0 1.333-.946 2.418-2.157 2.418z" />
            </svg>
            Discord
          </a>
          <Link
            href="/login"
            className="flex items-center gap-2 px-5 py-2 rounded-full text-black font-medium shadow-sm border-2 border-transparent hover:bg-transparent hover:border-[#b7ff00] hover:text-[#b7ff00] transition-colors text-sm"
            style={{ background: "#b7ff00" }}
          >
            <svg
              className="w-4 h-4"
              viewBox="0 0 24 24"
              fill="none"
              stroke="currentColor"
              strokeWidth="2"
              strokeLinecap="round"
              strokeLinejoin="round"
            >
              <rect x="2" y="3" width="20" height="14" rx="2" ry="2"></rect>
              <line x1="8" y1="21" x2="16" y2="21"></line>
              <line x1="12" y1="17" x2="12" y2="21"></line>
            </svg>
            Login
          </Link>
        </div>
      </div>
    </header>
  )
}
