import Image from "next/image"
import { Button } from "@/components/ui/button"
import { ExternalLink } from "lucide-react"

interface PortfolioCardProps {
  title: string
  description: string
  image: string
  badges?: string[]
  statusBadges?: { text: string; color: string }[]
  tags?: string[]
  isNew?: boolean
  demoLink?: string
  githubLink?: string
}

export function PortfolioCard({
  title,
  description,
  image,
  badges = [],
  statusBadges = [],
  isNew = false,
  demoLink = true,
}: PortfolioCardProps) {
  return (
    <div className="group relative rounded-xl overflow-hidden bg-card/50 backdrop-blur-sm border border-primary/20">
      <div className="absolute -inset-0.5 bg-gradient-to-r from-primary via-cyan-400 to-primary opacity-30 blur group-hover:opacity-50 transition-opacity duration-300" />

      <div className="relative bg-background/95 rounded-xl overflow-hidden">
        {/* Project Preview Image */}
        <div className="relative aspect-[16/9] overflow-hidden">
          <div className="absolute inset-0 bg-[linear-gradient(to_right,#3dd9b580_1px,transparent_1px),linear-gradient(to_bottom,#3dd9b580_1px,transparent_1px)] bg-[size:40px_40px] opacity-20 pointer-events-none z-10" />

          <Image
            src={image || "/placeholder.svg"}
            alt={title}
            fill
            className="object-cover transition-transform duration-500 group-hover:scale-110"
          />

          <div className="absolute inset-0 bg-gradient-to-t from-background via-background/50 to-transparent z-[5]" />

          <div className="absolute top-3 left-3 flex gap-2 z-20">
            {isNew && (
              <div className="px-3 py-1 bg-orange-500 text-white text-xs font-bold rounded-full flex items-center gap-1">
                <span className="w-1.5 h-1.5 bg-white rounded-full animate-pulse" />
                NEW
              </div>
            )}
            {badges.map((badge, i) => (
              <div
                key={i}
                className="px-3 py-1 bg-primary/90 backdrop-blur-sm text-primary-foreground text-xs font-bold rounded-full border border-primary"
              >
                {badge}
              </div>
            ))}
          </div>

          <div className="absolute bottom-0 left-0 right-0 p-4 z-20">
            <h3 className="text-2xl font-bold text-white drop-shadow-[0_2px_10px_rgba(0,0,0,0.8)]">{title}</h3>
          </div>
        </div>

        {/* Project Details */}
        <div className="p-5 space-y-4">
          <p className="text-sm text-muted-foreground line-clamp-2">{description}</p>

          {statusBadges.length > 0 && (
            <div className="flex flex-wrap gap-2">
              {statusBadges.map((badge, i) => (
                <div
                  key={i}
                  className={`px-3 py-1 text-xs font-bold rounded-md border ${
                    badge.color === "pink"
                      ? "bg-pink-500/20 text-pink-400 border-pink-500/50"
                      : badge.color === "red"
                        ? "bg-red-500/20 text-red-400 border-red-500/50"
                        : badge.color === "green"
                          ? "bg-green-500/20 text-green-400 border-green-500/50"
                          : "bg-primary/20 text-primary border-primary/50"
                  }`}
                >
                  {badge.text}
                </div>
              ))}
            </div>
          )}

          <div className="flex gap-3">
            {demoLink && (
              <Button className="w-full bg-primary hover:bg-primary/90 text-primary-foreground">
                <ExternalLink className="mr-2 h-4 w-4" />
                View Demo
              </Button>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}
